<?php
include '../database/db.php';
session_start();

// Error reporting
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', 'php_error.log');

// Prevent caching
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// CSRF token generation
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Redirect if not logged in
if (empty($_SESSION['status_Account']) || empty($_SESSION['email'])) {
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'error' => 'User not logged in.']);
        exit();
    }
    header("Location: ../index.php");
    exit();
}

// Fetch user_id
$email = $_SESSION['email'];
$stmt = $connection->prepare("SELECT user_id FROM data WHERE email = ?");
$stmt->bind_param("s", $email);
$stmt->execute();
$result = $stmt->get_result();
$user = $result->fetch_assoc();
$user_id = $user['user_id'] ?? null;
$stmt->close();

if (!$user_id) {
    header("Location: ../index.php");
    exit();
}

$errors = [];
$success_message = '';
$debug_log = [];

function logError($message) {
    file_put_contents('error_log.txt', date('Y-m-d H:i:s') . ": $message\n", FILE_APPEND);
}

// Directory paths
define('PHYSICAL_UPLOAD_DIR', '../ProfileImage/image/IdPhoto/');
define('RELATIVE_UPLOAD_DIR', '../ProfileImage/image/IdPhoto/');
define('PROFILE_PHYSICAL_UPLOAD_DIR', '../ProfileImage/image/Profile_Photo/');
define('PROFILE_RELATIVE_UPLOAD_DIR', '../ProfileImage/image/Profile_Photo/');

// List of valid regions
$valid_regions = [
    'Region I – Ilocos Region',
    'Region II – Cagayan Valley',
    'Region III – Central Luzon',
    'Region IV-A – CALABARZON',
    'MIMAROPA Region',
    'Region V – Bicol Region',
    'Region VI – Western Visayas',
    'Region VII – Central Visayas',
    'Region VIII – Eastern Visayas',
    'Region IX – Zamboanga Peninsula',
    'Region X – Northern Mindanao',
    'Region XI – Davao Region',
    'Region XII – SOCCSKSARGEN',
    'Region XIII – Caraga',
    'NCR – National Capital Region',
    'CAR – Cordillera Administrative Region',
    'BARMM – Bangsamoro Autonomous Region in Muslim Mindanao',
    'NIR – Negros Island Region'
];

// Function to generate a unique appointment ID
function generateAppointmentId($connection) {
    $maxRetries = 5;
    $attempt = 0;

    while ($attempt < $maxRetries) {
        $randomNumber = mt_rand(0, 99999999);
        $appointment_id = sprintf("%08d", $randomNumber);
        $stmt = $connection->prepare("SELECT COUNT(*) AS count FROM appointments WHERE appointment_id = ?");
        $stmt->bind_param("s", $appointment_id);
        $stmt->execute();
        $result = $stmt->get_result();
        $row = $result->fetch_assoc();
        $stmt->close();

        if ($row['count'] == 0) {
            return $appointment_id;
        }

        $attempt++;
    }

    throw new Exception("Unable to generate a unique appointment ID after $maxRetries attempts.");
}

// For AJAX requests, ensure proper JSON responses
if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
    header('Content-Type: application/json');
    
    // Set error handler for AJAX requests
    set_error_handler(function($errno, $errstr, $errfile, $errline) {
        echo json_encode([
            'success' => false,
            'error' => 'Server error occurred',
            'debug' => "$errstr in $errfile on line $errline"
        ]);
        exit;
    });
}

// Handle ID photo upload
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_FILES['idPhoto']) && isset($_POST['upload_id_photo'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $errors['general'] = 'Invalid CSRF token.';
        $debug_log[] = 'CSRF token validation failed.';
        logError('CSRF token validation failed for ID photo upload.');
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
            exit();
        }
    }

    $id_type = filter_input(INPUT_POST, 'idType', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $id_number = filter_input(INPUT_POST, 'idNumber', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $file = $_FILES['idPhoto'];
    $validTypes = ['image/jpeg', 'image/jpg'];

    $baseFileName = 'id_' . $user_id;
    $fileName = $baseFileName . '.jpg';
    $uploadPath = PHYSICAL_UPLOAD_DIR . $fileName;
    $relativePath = RELATIVE_UPLOAD_DIR . $fileName;

    $counter = 1;
    while (file_exists($uploadPath)) {
        $fileName = $baseFileName . '_' . $counter . '.jpg';
        $uploadPath = PHYSICAL_UPLOAD_DIR . $fileName;
        $relativePath = RELATIVE_UPLOAD_DIR . $fileName;
        $counter++;
    }

    if (!is_dir(PHYSICAL_UPLOAD_DIR)) {
        if (!mkdir(PHYSICAL_UPLOAD_DIR, 0777, true)) {
            $errors['idPhoto'] = 'Failed to create upload directory.';
            $debug_log[] = "Failed to create directory: " . PHYSICAL_UPLOAD_DIR;
            logError("Failed to create upload directory: " . PHYSICAL_UPLOAD_DIR);
        }
    }

    if (empty($id_type)) {
        $errors['idType'] = "Please select a valid ID type.";
    } elseif (empty($id_number) || !preg_match('/^[A-Za-z0-9-]{1,50}$/', $id_number)) {
        $errors['idNumber'] = "ID number must be alphanumeric (up to 50 characters).";
    } elseif (!in_array($file['type'], $validTypes)) {
        $errors['idPhoto'] = "Please upload a valid .jpg or .jpeg file.";
    } elseif ($file['error'] !== UPLOAD_ERR_OK) {
        $errors['idPhoto'] = "Upload failed: " . [
            UPLOAD_ERR_INI_SIZE => "File exceeds server size limit.",
            UPLOAD_ERR_FORM_SIZE => "File exceeds form size limit.",
            UPLOAD_ERR_PARTIAL => "File was only partially uploaded.",
            UPLOAD_ERR_NO_FILE => "No file was uploaded.",
            UPLOAD_ERR_NO_TMP_DIR => "Temporary folder missing.",
            UPLOAD_ERR_CANT_WRITE => "Failed to write file to disk.",
            UPLOAD_ERR_EXTENSION => "A PHP extension stopped the upload."
        ][$file['error']] ?? "Unknown error.";
    } elseif (!is_uploaded_file($file['tmp_name'])) {
        $errors['idPhoto'] = "Security error: Invalid file upload.";
    } else {
        $imageInfo = @getimagesize($file['tmp_name']);
        if ($imageInfo === false) {
            $errors['idPhoto'] = "Invalid image file. Please upload a valid JPEG.";
        } elseif ($imageInfo[0] < 180 || $imageInfo[1] < 180 || abs($imageInfo[0] - $imageInfo[1]) > 20) {
            $errors['idPhoto'] = "Image must be approximately 2x2 inches (~192x192 pixels at 96 DPI).";
        } else {
            if (extension_loaded('gd') && function_exists('imagecreatefromjpeg')) {
                $image = @imagecreatefromjpeg($file['tmp_name']);
                if ($image === false) {
                    $errors['idPhoto'] = "Failed to process image. Try another file.";
                } else {
                    if (imagejpeg($image, $uploadPath, 75)) {
                        imagedestroy($image);
                        $_SESSION['idPhoto'] = $relativePath;
                        $_SESSION['idType'] = $id_type;
                        $_SESSION['idNumber'] = $id_number;
                        $debug_log[] = "ID photo uploaded to: $uploadPath, stored as: $relativePath";
                        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                            header('Content-Type: application/json');
                            echo json_encode(['success' => true, 'message' => 'ID uploaded successfully.', 'idPhoto' => $relativePath]);
                            session_write_close();
                            exit();
                        }
                    } else {
                        imagedestroy($image);
                        $errors['idPhoto'] = "Failed to save image to server.";
                        $debug_log[] = "Failed to save image to: $uploadPath";
                        logError("Failed to save ID photo to: $uploadPath");
                    }
                }
            } else {
                if (!empty($_POST['compressedImage'])) {
                    $data = $_POST['compressedImage'];
                    $data = str_replace('data:image/jpeg;base64,', '', $data);
                    $data = base64_decode($data);
                    if ($data === false) {
                        $errors['idPhoto'] = "Invalid compressed image data.";
                    } elseif (file_put_contents($uploadPath, $data)) {
                        $_SESSION['idPhoto'] = $relativePath;
                        $_SESSION['idType'] = $id_type;
                        $_SESSION['idNumber'] = $id_number;
                        $debug_log[] = "Compressed ID photo uploaded to: $uploadPath, stored as: $relativePath";
                        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                            header('Content-Type: application/json');
                            echo json_encode(['success' => true, 'message' => 'ID uploaded successfully.', 'idPhoto' => $relativePath]);
                            session_write_close();
                            exit();
                        }
                    } else {
                        $errors['idPhoto'] = "Failed to save compressed image to server.";
                        $debug_log[] = "Failed to save compressed image to: $uploadPath";
                        logError("Failed to save compressed ID photo to: $uploadPath");
                    }
                } else {
                    $errors['idPhoto'] = "Server image processing unavailable.";
                }
            }
        }
    }

    if (!empty($errors) && !empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
        exit();
    }
}

// Handle photo removal
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['removeIdPhoto'])) {
    if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
        $errors['general'] = 'Invalid CSRF token.';
        $debug_log[] = 'CSRF token validation failed for photo removal.';
        logError('CSRF token validation failed for photo removal.');
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
            exit();
        }
    }

    if (!empty($_SESSION['idPhoto'])) {
        $fullPath = str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']);
        if (file_exists($fullPath)) {
            unlink($fullPath);
            $debug_log[] = "Removed ID photo: $fullPath";
        } else {
            $debug_log[] = "ID photo not found for removal: $fullPath";
        }
    }
    unset($_SESSION['idPhoto'], $_SESSION['idType'], $_SESSION['idNumber']);
    $debug_log[] = "Cleared ID photo session data.";
    if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
        header('Content-Type: application/json');
        echo json_encode(['success' => true, 'message' => 'ID photo removed successfully.']);
        session_write_close();
        exit();
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['submit_appointment'])) {
    try {
        if (!isset($_POST['csrf_token']) || $_POST['csrf_token'] !== $_SESSION['csrf_token']) {
            $errors['general'] = 'Invalid CSRF token.';
            $debug_log[] = 'CSRF token validation failed for form submission.';
            logError('CSRF token validation failed for form submission.');
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                header('Content-Type: application/json');
                echo json_encode(['success' => false, 'errors' => $errors, 'debug' => $debug_log]);
                exit();
            }
        }

        function sanitizeInput($input) {
            if (is_string($input)) {
                $input = strip_tags($input);
                $input = htmlspecialchars($input, ENT_QUOTES, 'UTF-8');
                return trim($input);
            }
            return $input;
        }

        $last_name = sanitizeInput($_POST['lastName'] ?? '');
        $first_name = sanitizeInput($_POST['firstName'] ?? '');
        $middle_name = !empty($_POST['middleName']) ? sanitizeInput($_POST['middleName']) : null;
        $gender = sanitizeInput($_POST['gender'] ?? '');
        $other_gender = !empty($_POST['othergender']) ? sanitizeInput($_POST['othergender']) : null;
        $birthdate = sanitizeInput($_POST['birthdate'] ?? '');
        $occupation = sanitizeInput($_POST['occupation'] ?? '');
      	$street = sanitizeInput($_POST['street'] ?? '');
      	$barangay = sanitizeInput($_POST['barangay'] ?? '');
      	$city = sanitizeInput($_POST['city'] ?? '');
      	$province = sanitizeInput($_POST['province'] ?? '');
        $address = sanitizeInput($_POST['address'] ?? '');
        $region = sanitizeInput($_POST['region'] ?? '');
        $contact = sanitizeInput($_POST['contact'] ?? '');
        $appointment_date = sanitizeInput($_POST['appointmentDate'] ?? '');
        $appointment_time = sanitizeInput($_POST['appointmentTime'] ?? '');
        $purpose = sanitizeInput($_POST['purpose'] ?? '');

        $email = $_SESSION['email']; // Use session email directly
        $profile_photo = $_SESSION['profilePhoto'] ?? '';
        $id_type = $_SESSION['idType'] ?? '';
        $id_number = $_SESSION['idNumber'] ?? '';
        $id_photo = $_SESSION['idPhoto'] ?? '';

        // Validate inputs
        if (empty($last_name)) $errors['lastName'] = 'Last name is required.';
        if (empty($first_name)) $errors['firstName'] = 'First name is required.';
        if (empty($gender)) $errors['gender'] = 'Gender is required.';
        if ($gender === 'Other' && empty($other_gender)) $errors['othergender'] = 'Please specify your gender.';
        if (empty($birthdate) || !DateTime::createFromFormat('Y-m-d', $birthdate)) $errors['birthdate'] = 'Valid date of birth is required.';
        if (empty($occupation)) $errors['occupation'] = 'Occupation is required.';
      	if (empty($street)) $errors['address'] = 'Street is required.';
      	if (empty($barangay)) $errors['address'] = 'Barangay is required.';
      	if (empty($city)) $errors['address'] = 'City/Municipality is required.';
      	if (empty($province)) $errors['address'] = 'Province is required.';
        if (empty($address)) $errors['address'] = 'Address is required.';
        if (empty($region) || !in_array($region, $valid_regions)) $errors['region'] = 'Please select a valid region.';
        if (empty($email) || !filter_var($email, FILTER_VALIDATE_EMAIL)) $errors['email'] = 'Valid email is required.';
        if (empty($contact) || !preg_match('/^[0-9]{10}$/', $contact)) $errors['contact'] = 'Valid 10-digit contact number is required.';
        if (empty($appointment_date)) $errors['appointmentDate'] = 'Appointment date is required.';
        if (empty($appointment_time)) {
            $errors['appointmentTime'] = 'Appointment time is required.';
        } elseif (!preg_match('/^([0-1][0-9]|2[0-2]):[0-5][0-9]$/', $appointment_time) || $appointment_time < '05:00' || $appointment_time > '22:00') {
            $errors['appointmentTime'] = 'Appointment time must be between 5:00 AM and 10:00 PM.';
            $debug_log[] = "Invalid appointment time: $appointment_time";
            logError("Invalid appointment time: $appointment_time");
        }
        if (empty($purpose)) $errors['purpose'] = 'Purpose is required.';
        if (empty($profile_photo)) {
            $errors['myFile'] = 'Profile photo is required.';
            $debug_log[] = "Profile photo missing: $profile_photo";
        } elseif (!file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo))) {
            $errors['myFile'] = 'Profile photo file not found.';
            $debug_log[] = "Profile photo not found: " . str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo);
            logError("Profile photo not found: " . str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $profile_photo));
        }
        if (empty($id_type)) $errors['idType'] = 'ID type is required.';
        if (empty($id_number)) $errors['idNumber'] = 'ID number is required.';
        if (empty($id_photo)) {
            $errors['idPhoto'] = 'ID photo is required.';
            $debug_log[] = "ID photo missing: $id_photo";
        } elseif (!file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo))) {
            $errors['idPhoto'] = 'ID photo file not found.';
            $debug_log[] = "ID photo not found: " . str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo);
            logError("ID photo not found: " . str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $id_photo));
        }

        if (!empty($appointment_date)) {
            $date = DateTime::createFromFormat('Y-m-d', $appointment_date);
            if ($date === false || $date->format('N') !== '4') {
                $errors['appointmentDate'] = 'Appointment date must be a Thursday.';
            }
        }

        if (!empty($errors)) {
            if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
                echo json_encode([
                    'success' => false,
                    'errors' => $errors,
                    'debug' => $debug_log
                ]);
                exit;
            }
        } else {
            // Database operations...
            $connection->begin_transaction();
            try {
                // Generate unique appointment ID
                $appointment_id = generateAppointmentId($connection);

                // Calculate age from birthdate
                $birthdate_obj = new DateTime($birthdate);
                $today = new DateTime();
                $age = $birthdate_obj->diff($today)->y;

                // Prepare user information statement
                $userStmt = $connection->prepare("
                    INSERT INTO user_information (
                        user_id, last_name, first_name, middle_name, gender, other_gender,
                        birthdate, age, occupation, address, region, email, contact,
                        profile_photo, id_type, id_number, id_photo
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
                    ON DUPLICATE KEY UPDATE
                        last_name = VALUES(last_name),
                        first_name = VALUES(first_name),
                        middle_name = VALUES(middle_name),
                        gender = VALUES(gender),
                        other_gender = VALUES(other_gender),
                        birthdate = VALUES(birthdate),
                        age = VALUES(age),
                        occupation = VALUES(occupation),
                        address = VALUES(address),
                        region = VALUES(region),
                        contact = VALUES(contact),
                        profile_photo = VALUES(profile_photo),
                        id_type = VALUES(id_type),
                        id_number = VALUES(id_number),
                        id_photo = VALUES(id_photo)
                ");

                if (!$userStmt) {
                    throw new Exception("Prepare failed: " . $connection->error);
                }

                $userStmt->bind_param(
                    "sssssssssssssssss",
                    $user_id, $last_name, $first_name, $middle_name, $gender, $other_gender,
                    $birthdate, $age, $occupation, $address, $region, $email, $contact,
                    $profile_photo, $id_type, $id_number, $id_photo
                );

                if (!$userStmt->execute()) {
                    throw new Exception("Error updating user information: " . $userStmt->error);
                }

                // Prepare appointments statement
                $apptStmt = $connection->prepare("
                    INSERT INTO appointments (
                        appointment_id, user_id, last_name, first_name, middle_name,
                        gender, other_gender, birthdate, age, occupation, address,
                        region, email, contact, appointment_date, appointment_time,
                        purpose, profile_photo, id_type, id_number, id_photo, status
                    ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')
                ");

                if (!$apptStmt) {
                    throw new Exception("Prepare failed: " . $connection->error);
                }

                $apptStmt->bind_param(
                    "sissssssissssssssssss",
                    $appointment_id, $user_id, $last_name, $first_name, $middle_name,
                    $gender, $other_gender, $birthdate, $age, $occupation, $address,
                    $region, $email, $contact, $appointment_date, $appointment_time,
                    $purpose, $profile_photo, $id_type, $id_number, $id_photo
                );

                if (!$apptStmt->execute()) {
                    throw new Exception("Error inserting appointment: " . $apptStmt->error);
                }

                $connection->commit();
                
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => true,
                    'message' => "Appointment submitted successfully with ID: $appointment_id"
                ]);
                exit();

            } catch (Exception $e) {
                $connection->rollback();
                error_log("Database error: " . $e->getMessage());
                
                header('Content-Type: application/json');
                echo json_encode([
                    'success' => false,
                    'error' => 'Database error occurred. Please try again.',
                    'debug' => $e->getMessage()
                ]);
                exit();
            }
        }
    } catch (Exception $e) {
        if (!empty($_SERVER['HTTP_X_REQUESTED_WITH']) && strtolower($_SERVER['HTTP_X_REQUESTED_WITH']) == 'xmlhttprequest') {
            header('Content-Type: application/json');
            echo json_encode([
                'success' => false,
                'error' => 'Server error: ' . $e->getMessage(),
                'debug' => $debug_log
            ]);
            exit;
        }
        throw $e;
    }
}

$connection->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css" media="all">
    <link rel="icon" type="image/x-icon" href="../image/icons/logo1.ico">
    <title>Registration Form</title>
</head>
<body>
    <header class="form-header">
            <img src="../image/icons/logo1.ico" alt="Organization Logo" class="logo">
            <h1 id="form-title">Registration Form</h1>
            <a href="../logout.php" class="logout-link">Logout</a>
    </header>
    <div class="form-container">
        <form id="registrationForm" action="" method="POST" enctype="multipart/form-data" aria-label="Registration Form">
            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($_SESSION['csrf_token']); ?>">

            <!-- Photo Upload Section -->
            <div class="form-section">
                <h2>Photo Identification</h2>
                <div class="form-group">
                    <label for="myFile" class="label required">Profile Photo (2x2, JPG/JPEG)</label>
                    <div class="photo-upload-group">
                        <div class="photo-placeholder" id="profilePhotoPreview" aria-live="polite">
                            <img id="profilePhotoImg" class="photo-preview <?php echo !empty($_SESSION['profilePhoto']) && file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto'])) ? 'active' : ''; ?>" src="<?php echo htmlspecialchars($_SESSION['profilePhoto'] ?? ''); ?>" alt="Profile Photo Preview" aria-hidden="<?php echo empty($_SESSION['profilePhoto']) || !file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto'])) ? 'true' : 'false'; ?>">
                            <?php if (empty($_SESSION['profilePhoto']) || !file_exists(str_replace(PROFILE_RELATIVE_UPLOAD_DIR, PROFILE_PHYSICAL_UPLOAD_DIR, $_SESSION['profilePhoto']))): ?>
                                <span>No Photo Uploaded</span>
                            <?php endif; ?>
                        </div>
                        <a href="../photo_config_upload/photo_upload.php" class="btn btn-primary" aria-label="Upload profile photo">Upload Profile Photo</a>
                        <div class="photo-upload-note">JPG/JPEG, Max 2MB, 192x192 pixels</div>
                    </div>
                    <span class="error" id="myFile-error"><?php echo htmlspecialchars($errors['myFile'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="idType" class="label required">Valid ID Type</label>
                        <select id="idType" name="idType" required aria-required="true">
                            <option value="" <?php echo empty($_SESSION['idType']) ? 'selected' : ''; ?>>Select ID Type</option>
                            <?php
                            $idTypes = [
                                "Professional Regulation Commission", "Government Service Insurance System", "Passport", "SSS ID", "Drivers License",
                                "Overseas Workers Welfare Administration", "Senior Citizen ID", "NBI Clearance", "Unified Multi-purpose Identification (UMID) Card",
                                "Voters ID", "TIN ID", "PhilHealth ID", "Postal ID", "Seamans Book", "Philippine Identification Card",
                                "Philippine Passport", "Philippine Postal ID", "Police Clearance", "Barangay Clearance", "Integrated Bar of the Philippines",
                                "National ID", "Philippine Identification (PhilID)/ePhilID", "School ID", "Alien Certification"
                            ];
                            foreach ($idTypes as $type) {
                                $selected = (isset($_SESSION['idType']) && $_SESSION['idType'] === $type) ? 'selected' : '';
                                echo "<option value=\"" . htmlspecialchars($type) . "\" $selected>" . htmlspecialchars($type) . "</option>";
                            }
                            ?>
                        </select>
                        <span class="error" id="idType-error"><?php echo htmlspecialchars($errors['idType'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="idNumber" class="label required">ID Number</label>
                        <input type="text" id="idNumber" name="idNumber" required autocomplete="off" aria-required="true" value="<?php echo htmlspecialchars($_SESSION['idNumber'] ?? ''); ?>" placeholder="Example: xxxx-xxxx-xxxx-xxxx" maxlength="50">
                        <span class="error" id="idNumber-error"><?php echo htmlspecialchars($errors['idNumber'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="idPhoto" class="label required">ID Photo (2x2, JPG/JPEG)</label>
                    <div class="photo-upload-group">
                        <div class="photo-placeholder" id="idPhotoPreview" aria-live="polite">
                            <img id="idPhotoImg" class="photo-preview <?php echo !empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'active' : ''; ?>" src="<?php echo !empty($_SESSION['idPhoto']) ? htmlspecialchars($_SESSION['idPhoto']) : ''; ?>" alt="ID Photo Preview" aria-hidden="<?php echo empty($_SESSION['idPhoto']) || !file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'true' : 'false'; ?>">
                            <?php if (empty($_SESSION['idPhoto']) || !file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']))): ?>
                                <span>No ID Photo Uploaded</span>
                            <?php endif; ?>
                        </div>
                        <div class="action-buttons" id="actionButtons">
                            <button type="button" class="btn btn-primary" id="chooseIdPhotoBtn" aria-label="Choose ID photo file">Select ID Photo</button>
                            <?php if (!empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto']))): ?>
                                <button type="button" class="btn btn-danger" id="removeIdPhotoBtn" aria-label="Remove uploaded ID photo">Remove ID Photo</button>
                            <?php endif; ?>
                        </div>
                        <input type="file" id="idPhotoInput" name="idPhoto" accept="image/jpeg,image/jpg" style="display: none;" aria-label="Select ID photo">
                        <input type="hidden" id="compressedImage" name="compressedImage">
                        <div class="photo-upload-note">JPG/JPEG, Max 2MB, 192x192 pixels. Photo uploads automatically after selection.</div>
                        <div class="progress-bar" id="idPhotoProgress"><div></div></div>
                        <div class="loading-spinner" id="idPhotoSpinner" aria-label="Processing"></div>
                        <div class="success-message" id="idPhotoSuccess" role="alert"><?php echo !empty($_SESSION['idPhoto']) && file_exists(str_replace(RELATIVE_UPLOAD_DIR, PHYSICAL_UPLOAD_DIR, $_SESSION['idPhoto'])) ? 'ID uploaded successfully.' : ''; ?></div>
                    </div>
                    <span class="error" id="idPhoto-error"><?php echo htmlspecialchars($errors['idPhoto'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Personal Information Section -->
            <div class="form-section">
                <h2>Personal Information</h2>
                <div class="form-row">
                    <div class="form-group">
                        <label for="lastName" class="label required">Last Name</label>
                        <input type="text" id="lastName" name="lastName" required autocomplete="off" aria-required="true" placeholder="Enter your last name">
                        <span class="error" id="lastName-error"><?php echo htmlspecialchars($errors['lastName'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="firstName" class="label required">First Name</label>
                        <input type="text" id="firstName" name="firstName" required autocomplete="off" aria-required="true" placeholder="Enter your first name">
                        <span class="error" id="firstName-error"><?php echo htmlspecialchars($errors['firstName'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="middleName" class="label">Middle Name (Optional)</label>
                    <input type="text" id="middleName" name="middleName" autocomplete="off" placeholder="Enter your middle name">
                    <span class="error" id="middleName-error"><?php echo htmlspecialchars($errors['middleName'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="gender" class="label required">Gender</label>
                        <select id="gender" name="gender" required aria-required="true">
                            <option value="" selected disabled>Select Gender</option>
                            <option value="Male">Male</option>
                            <option value="Female">Female</option>
                            <option value="Other">Other</option>
                        </select>
                        <span class="error" id="gender-error"><?php echo htmlspecialchars($errors['gender'] ?? ''); ?></span>
                    </div>
                    <div class="form-group" id="othergenderGroup" style="display: none;">
                        <label for="othergender" class="label required">Specify Gender</label>
                        <input type="text" id="othergender" name="othergender" autocomplete="off" placeholder="Specify your gender">
                        <span class="error" id="othergender-error"><?php echo htmlspecialchars($errors['othergender'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="birthdate" class="label required">Date of Birth</label>
                        <input type="date" id="birthdate" name="birthdate" required autocomplete="off" max="<?php echo date('Y-m-d'); ?>" aria-required="true">
                        <span class="error" id="birthdate-error"><?php echo htmlspecialchars($errors['birthdate'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="age" class="label required">Age</label>
                        <input type="number" id="age" name="age" required autocomplete="off" min="1" max="120" aria-required="true" placeholder="Auto-calculated">
                        <span class="error" id="age-error"><?php echo htmlspecialchars($errors['age'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="occupation" class="label required">Occupation</label>
                    <input type="text" id="occupation" name="occupation" required autocomplete="off" aria-required="true" placeholder="Enter your occupation">
                    <span class="error" id="occupation-error"><?php echo htmlspecialchars($errors['occupation'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Contact and Address Section -->
            <div class="form-section">
                <h2>Contact and Address</h2>
                <div class="form-group">
                      <label for="street" class="label required">Street</label>
                    <input type="text" id="street" name="street">
                    <label for="barangay" class="label required">Barangay</label>
                    <input type="text" id="barangay" name="barangay">
                    <label for="city" class="label required">City/Municipality</label>
                    <input type="text" id="city" name="city">
                    <label for="province" class="label required">Province</label>
                    <input type="text" id="province" name="province">
                    <label for="address" class="label required">Full Address</label>
                    <input type="text" id="address" name="address" required autocomplete="off" aria-required="true" readonly>
                    <span class="error" id="address-error"><?php echo htmlspecialchars($errors['address'] ?? ''); ?></span>
                </div>
              <div class="form-group">
                    <label for="region" class="label required">Region</label>
                    <select id="region" name="region" required aria-required="true" aria-describedby="region-help">
                        <option value="" selected disabled>Select a Region</option>
                        <?php foreach ($valid_regions as $region_option): ?>
                            <option value="<?php echo htmlspecialchars($region_option); ?>">
                                <?php echo htmlspecialchars($region_option); ?>
                            </option>
                        <?php endforeach; ?>
                    </select>
                    <div class="photo-upload-note">Please select your region from the list.</div>
                    <span class="error" id="region-error"><?php echo htmlspecialchars($errors['region'] ?? ''); ?></span>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label for="email" class="label required">Email Address</label>
                        <input type="email" id="email" name="email" required readonly autocomplete="off" aria-required="true" value="<?php echo htmlspecialchars($_SESSION['email']); ?>">
                        <span class="error" id="email-error"><?php echo htmlspecialchars($errors['email'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="contact" class="label required">Contact Number</label>
                        <div class="contact-group">
                            <input type="text" value="+63" readonly aria-label="Country code">
                            <input type="text" id="contact" name="contact" required autocomplete="off" pattern="[0-9]{10}" title="Please enter a valid 10-digit phone number" placeholder="9123456789" aria-required="true">
                        </div>
                        <span class="error" id="contact-error"><?php echo htmlspecialchars($errors['contact'] ?? ''); ?></span>
                    </div>
                </div>
            </div>

            <!-- Appointment Details Section -->
            <div class="form-section">
                <h2>Appointment Details</h2>
                <div class="form-row">
                    <div class="form-group">
                        <label for="appointmentDate" class="label required">Appointment Date (Thursdays)</label>
                        <input type="text" id="appointmentDate" name="appointmentDate" required autocomplete="off" aria-required="true" placeholder="Select a Thursday">
                        <div class="photo-upload-note">Only Thursdays are selectable</div>
                        <span class="error" id="appointmentDate-error"><?php echo htmlspecialchars($errors['appointmentDate'] ?? ''); ?></span>
                    </div>
                    <div class="form-group">
                        <label for="appointmentTime" class="label required">Appointment Time (Fixed at 10:00 AM)</label>
                        <input type="time" id="appointmentTime" name="appointmentTime" required autocomplete="off" min="05:00" max="22:00" value="10:00" readonly aria-required="true">
                        <div class="photo-upload-note">Fixed at 10:00 AM</div>
                        <span class="error" id="appointmentTime-error"><?php echo htmlspecialchars($errors['appointmentTime'] ?? ''); ?></span>
                    </div>
                </div>
                <div class="form-group">
                    <label for="purpose" class="label required">Purpose of Appointment</label>
                    <input type="text" id="purpose" name="purpose" required readonly autocomplete="off" aria-required="true" value="Orientation" placeholder="Enter the purpose of your appointment">
                    <span class="error" id="purpose-error"><?php echo htmlspecialchars($errors['purpose'] ?? ''); ?></span>
                </div>
            </div>

            <!-- Feedback and Submit -->
            <div class="form-group">
                <div class="success-message" id="successMessage" role="alert"><?php echo htmlspecialchars($success_message); ?></div>
                <div class="error-message" id="errorMessage" role="alert"><?php echo htmlspecialchars($errors['general'] ?? ''); ?></div>
                <div class="loading-spinner" id="loadingSpinner" aria-label="Loading"></div>
            </div>
            <?php if (!empty($debug_log)): ?>
                <div class="debug-log">
                    <h3>Debug Log:</h3>
                    <?php foreach ($debug_log as $log): ?>
                        <p><?php echo htmlspecialchars($log); ?></p>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
            <div class="form-group" style="text-align: center;">
                <button type="submit" name="submit_appointment" value="1" class="btn btn-primary" aria-label="Submit appointment form">Submit Appointment</button>
            </div>
        </form>
        </div>

    <script src="https://cdn.jsdelivr.net/npm/flatpickr" defer></script>
</body>
  <script>
const mergeAddress = () => {
    const street = document.getElementById('street').value.trim();
    const barangay = document.getElementById('barangay').value.trim();
    const city = document.getElementById('city').value.trim();
    const province = document.getElementById('province').value.trim();

    document.getElementById('address').value = [street, barangay, city, province].filter(v => v).join(', ');

    if (!street || !barangay || !city || !province) {
        document.getElementById('address-error').textContent = 'Please fill in all address fields.';
    } else {
        document.getElementById('address-error').textContent = '';
    }
};

['street', 'barangay', 'city', 'province'].forEach(id => {
    document.getElementById(id).addEventListener('input', mergeAddress);
});

//flatpicker
const MAX_APPOINTMENTS = 20;
let appointmentCounts = {};

fetch('./get_appointment_counts.php')
    .then(response => response.json())
    .then(data => {
        appointmentCounts = data;
        initFlatpickr();
    })
    .catch(error => console.error(error));

function initFlatpickr() {
    flatpickr('#appointmentDate', {
        dateFormat: 'Y-m-d',
        minDate: 'today', // optional, ensures past dates disabled

        disable: [
            function(date) {
                const today = new Date();
                today.setHours(0, 0, 0, 0); // midnight for proper comparison
                const formatted = flatpickr.formatDate(date, 'Y-m-d');
                const count = appointmentCounts?.[formatted] ?? 0;

                // Disable if:
                // 1. Not Thursday
                // 2. Past date
                // 3. Fully booked
                return date.getDay() !== 4 || date < today || count >= MAX_APPOINTMENTS;
            }
        ],

        onDayCreate: (dObj, dStr, fp, dayElem) => {
            const date = flatpickr.formatDate(dayElem.dateObj, 'Y-m-d');
            const count = appointmentCounts?.[date] ?? 0;
            const today = new Date();
            today.setHours(0,0,0,0);

            if (count >= MAX_APPOINTMENTS) {
                dayElem.classList.add('full-date');
            } else if (dayElem.dateObj.getDay() === 4 && dayElem.dateObj >= today) {
                dayElem.classList.add('thursday', 'available-date');
            } else {
                dayElem.classList.add('disabled-date'); // past or non-Thursday
            }
        }
    });
}

  // Utility Functions
        const debounce = (func, wait) => {
            let timeout;
            return (...args) => {
                clearTimeout(timeout);
                timeout = setTimeout(() => func.apply(this, args), wait);
            };
        };

        const showError = (id, message) => {
            const errorEl = document.getElementById(`${id}-error`);
            if (errorEl) {
                errorEl.textContent = message;
                errorEl.style.display = message ? 'block' : 'none';
                errorEl.setAttribute('aria-live', 'assertive');
            }
        };

        const showSuccess = (id, message) => {
            const successEl = document.getElementById(id);
            if (successEl) {
                successEl.textContent = message;
                successEl.style.display = message ? 'block' : 'none';
                successEl.setAttribute('aria-live', 'assertive');
            }
        };

        // Main Logic
        document.addEventListener('DOMContentLoaded', () => {
            const form = document.getElementById('registrationForm');
            const elements = {
                gender: document.getElementById('gender'),
                othergenderGroup: document.getElementById('othergenderGroup'),
                birthdate: document.getElementById('birthdate'),
                age: document.getElementById('age'),
                contact: document.getElementById('contact'),
                appointmentDate: document.getElementById('appointmentDate'),
                appointmentTime: document.getElementById('appointmentTime'),
                successMessage: document.getElementById('successMessage'),
                errorMessage: document.getElementById('errorMessage'),
                loadingSpinner: document.getElementById('loadingSpinner'),
                idType: document.getElementById('idType'),
                idNumber: document.getElementById('idNumber'),
                idPhotoInput: document.getElementById('idPhotoInput'),
                idPhotoPreview: document.getElementById('idPhotoImg'),
                chooseIdPhotoBtn: document.getElementById('chooseIdPhotoBtn'),
                removeIdPhotoBtn: document.getElementById('removeIdPhotoBtn'),
                idPhotoSpinner: document.getElementById('idPhotoSpinner'),
                idPhotoSuccess: document.getElementById('idPhotoSuccess'),
                idPhotoProgress: document.getElementById('idPhotoProgress'),
                compressedImage: document.getElementById('compressedImage'),
                actionButtons: document.getElementById('actionButtons'),
                region: document.getElementById('region'),
                email: document.getElementById('email'),
                csrfToken: document.querySelector('input[name="csrf_token"]'), // <-- get CSRF token from hidden input
                profilePhoto: document.getElementById('profilePhotoImg'), // get profile photo src
                idPhoto: document.getElementById('idPhotoImg') // get id photo src
            };

            // Initialize debug log visibility
            const debugLog = document.querySelector('.debug-log');
            if (debugLog && debugLog.textContent.trim()) {
                debugLog.style.display = 'block';
            }



            // Toggle Other Gender
            const toggleOtherGender = () => {
                elements.othergenderGroup.style.display = elements.gender.value === 'Other' ? 'block' : 'none';
            };
            elements.gender.addEventListener('change', toggleOtherGender);
            toggleOtherGender();

            // Auto-calculate Age
            elements.birthdate.addEventListener('change', () => {
                const birthdate = new Date(elements.birthdate.value);
                const today = new Date();
                let age = today.getFullYear() - birthdate.getFullYear();
                const monthDiff = today.getMonth() - birthdate.getMonth();
                if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthdate.getDate())) {
                    age--;
                }
                elements.age.value = age >= 0 ? age : '';
                validateField('age', elements.age.value);
            });

            // Restrict Contact Input
            elements.contact.addEventListener('input', e => {
                e.target.value = e.target.value.replace(/[^0-9]/g, '').slice(0, 10);
                validateField('contact', e.target.value);
            });

            // Image Compression
            const compressImage = (file, maxSizeMB, targetSize, callback) => {
                const maxSizeBytes = maxSizeMB * 1024 * 1024;
                elements.idPhotoSpinner.style.display = 'block';
                elements.idPhotoProgress.style.display = 'block';
                elements.idPhotoProgress.querySelector('div').style.width = '10%';

                const img = new Image();
                const reader = new FileReader();
                reader.onload = e => {
                    img.src = e.target.result;
                    img.onload = () => {
                        try {
                            const canvas = document.createElement('canvas');
                            const ctx = canvas.getContext('2d');
                            const size = Math.min(img.width, img.height);
                            canvas.width = targetSize;
                            canvas.height = targetSize;
                            const offsetX = (img.width - size) / 2;
                            const offsetY = (img.height - size) / 2;
                            ctx.drawImage(img, offsetX, offsetY, size, size, 0, 0, targetSize, targetSize);

                            let quality = 0.85;
                            let compressedDataUrl;
                            do {
                                compressedDataUrl = canvas.toDataURL('image/jpeg', quality);
                                quality -= 0.05;
                                elements.idPhotoProgress.querySelector('div').style.width = `${100 - (quality * 100)}%`;
                            } while (compressedDataUrl.length / 4 * 3 > maxSizeBytes && quality > 0.1);

                            fetch(compressedDataUrl)
                                .then(res => res.blob())
                                .then(blob => {
                                    const compressedFile = new File([blob], file.name, { type: 'image/jpeg' });
                                    callback(compressedFile, compressedDataUrl);
                                    elements.idPhotoSpinner.style.display = 'none';
                                    elements.idPhotoProgress.style.display = 'none';
                                })
                                .catch(() => {
                                    showError('idPhoto', 'Error compressing image.');
                                    elements.idPhotoSpinner.style.display = 'none';
                                    elements.idPhotoProgress.style.display = 'none';
                                });
                        } catch (error) {
                            console.error('Image compression error:', error);
                            showError('idPhoto', 'Error processing image.');
                            elements.idPhotoSpinner.style.display = 'none';
                            elements.idPhotoProgress.style.display = 'none';
                        }
                    };
                    img.onerror = () => {
                        showError('idPhoto', 'Invalid image file. Please upload a valid JPEG.');
                        elements.idPhotoSpinner.style.display = 'none';
                        elements.idPhotoProgress.style.display = 'none';
                    };
                };
                reader.onerror = () => {
                    showError('idPhoto', 'Error reading file.');
                    elements.idPhotoSpinner.style.display = 'none';
                    elements.idPhotoProgress.style.display = 'none';
                };
                reader.readAsDataURL(file);
            };

            // Add Remove ID Photo Button
            const addRemoveButton = () => {
                const existingBtn = document.getElementById('removeIdPhotoBtn');
                if (existingBtn) {
                    existingBtn.remove();
                }

                const btn = document.createElement('button');
                btn.type = 'button';
                btn.className = 'btn btn-danger';
                btn.id = 'removeIdPhotoBtn';
                btn.textContent = 'Remove ID Photo';
                btn.setAttribute('aria-label', 'Remove uploaded ID photo');
                elements.actionButtons.appendChild(btn);
                btn.addEventListener('click', handleRemoveIdPhoto);
                elements.removeIdPhotoBtn = btn;
            };

            // ID Photo Selection and Auto-Upload
            elements.chooseIdPhotoBtn.addEventListener('click', () => elements.idPhotoInput.click());

            elements.idPhotoInput.addEventListener('change', e => {
                const file = e.target.files[0];
                showError('idPhoto', '');
                showSuccess('idPhotoSuccess', '');

                if (file) {
                    if (!['image/jpeg', 'image/jpg'].includes(file.type)) {
                        showError('idPhoto', 'Please upload a .jpg or .jpeg file.');
                        return;
                    }

                    if (!elements.idType.value) {
                        showError('idType', 'Please select a valid ID type.');
                        return;
                    }
                    if (!elements.idNumber.value || !/^[A-Za-z0-9-]{1,50}$/.test(elements.idNumber.value)) {
                        showError('idNumber', 'ID number must be alphanumeric (up to 50 characters).');
                        return;
                    }

                    compressImage(file, 0.3, 192, (compressedFile, dataUrl) => {
                        elements.idPhotoPreview.src = dataUrl;
                        elements.idPhotoPreview.classList.add('active');
                        elements.idPhotoPreview.setAttribute('aria-hidden', 'false');

                        const dataTransfer = new DataTransfer();
                        dataTransfer.items.add(compressedFile);
                        elements.idPhotoInput.files = dataTransfer.files;
                        elements.compressedImage.value = dataUrl;

                        // Auto-upload
                        const formData = new FormData(form);
                        formData.append('upload_id_photo', '1');
                        formData.append('csrf_token', elements.csrfToken.value); // <-- add CSRF token

                        elements.idPhotoSpinner.style.display = 'block';
                        fetch('', {
                            method: 'POST',
                            body: formData,
                            headers: { 'X-Requested-With': 'XMLHttpRequest' }
                        })
                            .then(response => response.json())
                            .then(data => {
                                elements.idPhotoSpinner.style.display = 'none';
                                if (data.success) {
                                    showSuccess('idPhotoSuccess', data.message);
                                    elements.idPhotoPreview.src = data.idPhoto;
                                    elements.idPhotoPreview.classList.add('active');
                                    elements.idPhotoPreview.setAttribute('aria-hidden', 'false');
                                    addRemoveButton();
                                    // Auto-refresh after successful upload
                                    setTimeout(() => {
                                        window.location.reload();
                                    }, 1000);
                                } else {
                                    showError('idPhoto', data.errors?.idPhoto || 'Failed to upload ID.');
                                    if (data.errors?.idType) showError('idType', data.errors.idType);
                                    if (data.errors?.idNumber) showError('idNumber', data.errors.idNumber);
                                    elements.idPhotoPreview.src = '';
                                    elements.idPhotoPreview.classList.remove('active');
                                    elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                                    elements.idPhotoInput.value = '';
                                    elements.compressedImage.value = '';
                                }
                            })
                            .catch(error => {
                                console.error('Upload error:', error);
                                elements.idPhotoSpinner.style.display = 'none';
                                showError('idPhoto', 'Network error.');
                                elements.idPhotoPreview.src = '';
                                elements.idPhotoPreview.classList.remove('active');
                                elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                                elements.idPhotoInput.value = '';
                                elements.compressedImage.value = '';
                            });
                    });
                }
            });

            // Remove ID Photo
            const handleRemoveIdPhoto = () => {
                const formData = new FormData();
                formData.append('removeIdPhoto', 'true');
                formData.append('csrf_token', elements.csrfToken.value); // <-- add CSRF token
                elements.idPhotoSpinner.style.display = 'block';
                fetch('', {
                    method: 'POST',
                    body: formData,
                    headers: { 'X-Requested-With': 'XMLHttpRequest' }
                })
                    .then(response => response.json())
                    .then(data => {
                        elements.idPhotoSpinner.style.display = 'none';
                        if (data.success) {
                            showSuccess('idPhotoSuccess', data.message);
                            elements.idPhotoPreview.src = '';
                            elements.idPhotoPreview.classList.remove('active');
                            elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                            elements.idPhotoInput.value = '';
                            elements.compressedImage.value = '';
                            elements.idType.value = '';
                            elements.idNumber.value = '';
                            const removeBtn = document.getElementById('removeIdPhotoBtn');
                            if (removeBtn) removeBtn.remove();
                            elements.removeIdPhotoBtn = null;
                        } else {
                            showError('idPhoto', 'Error removing ID photo.');
                        }
                    })
                    .catch(error => {
                        console.error('Remove photo error:', error);
                        elements.idPhotoSpinner.style.display = 'none';
                        showError('idPhoto', 'Network error.');
                    });
            };

            if (elements.removeIdPhotoBtn) {
                elements.removeIdPhotoBtn.addEventListener('click', handleRemoveIdPhoto);
            }

            // Real-time Validation
            const validateField = (id, value) => {
                const fieldLabels = {
                    lastName: 'Last name',
                    firstName: 'First name',
                    occupation: 'Occupation',
                    address: 'Address',
                    purpose: 'Purpose',
                    region: 'Region',
                    email: 'Email',
                    contact: 'Contact number',
                    age: 'Age',
                    idNumber: 'ID number',
                    gender: 'Gender',
                    idType: 'ID type'
                };

                try {
                    switch (id) {
                        case 'lastName':
                        case 'firstName':
                        case 'occupation':
                        case 'address':
                        case 'purpose':
                            showError(id, value.trim() ? '' : `${fieldLabels[id]} is required.`);
                            break;
                        case 'region':
                            showError(id, value.trim() ? '' : 'Please select a valid region.');
                            break;
                        case 'email':
                            showError(id, /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value) ? '' : 'Please enter a valid email.');
                            break;
                        case 'contact':
                            showError(id, /^[0-9]{10}$/.test(value) ? '' : 'Please enter a valid 10-digit number.');
                            break;
                        case 'age':
                            showError(id, (value >= 1 && value <= 120) ? '' : 'Age must be between 1 and 120.');
                            break;
                        case 'idNumber':
                            showError(id, /^[A-Za-z0-9-]{1,50}$/.test(value) ? '' : 'ID number must be alphanumeric (up to 50 characters).');
                            break;
                        case 'gender':
                            showError(id, value ? '' : 'Please select a gender.');
                            if (value === 'Other') {
                                const otherValue = document.getElementById('othergender').value;
                                showError('othergender', otherValue.trim() ? '' : 'Please specify your gender.');
                            }
                            break;
                        case 'idType':
                            showError(id, value ? '' : 'Please select a valid ID type.');
                            break;
                    }
                } catch (error) {
                    console.error(`Validation error for ${id}:`, error);
                    showError(id, 'Validation error occurred.');
                }
            };

            ['lastName', 'firstName', 'occupation', 'address', 'region', 'purpose', 'contact', 'age', 'idNumber', 'gender', 'idType'].forEach(id => {
                const input = document.getElementById(id);
                if (input) {
                    input.addEventListener('input', debounce(e => validateField(id, e.target.value), 300));
                }
            });

            // Form Validation
            const validateForm = () => {
                let isValid = true;
                const fields = [
                    { id: 'lastName', message: 'Last name is required.' },
                    { id: 'firstName', message: 'First name is required.' },
                    { id: 'gender', message: 'Please select a gender.' },
                    { id: 'birthdate', message: 'Date of birth is required.' },
                    { id: 'age', message: 'Age is required.' },
                    { id: 'occupation', message: 'Occupation is required.' },
                    { id: 'address', message: 'Address is required.' },
                    { id: 'region', message: 'Please select a valid region.' },
                    { id: 'email', message: 'Email is required.' },
                    { id: 'contact', message: 'Contact number is required.' },
                    { id: 'purpose', message: 'Purpose is required.' },
                    { id: 'idType', message: 'ID type is required.' },
                    { id: 'idNumber', message: 'ID number is required.' }
                ];

                try {
                    // Clear all previous errors first
                    fields.forEach(field => {
                        showError(field.id, '');
                    });
                    showError('appointmentDate', '');
                    showError('errorMessage', '');

                    // Validate all fields
                    fields.forEach(field => {
                        const input = document.getElementById(field.id);
                        if (!input || !input.value.trim()) {
                            showError(field.id, field.message);
                            isValid = false;
                        } else {
                            validateField(field.id, input.value);
                            if (document.getElementById(`${field.id}-error`).style.display === 'block') {
                                isValid = false;
                            }
                        }
                    });

                    // Special validation for appointment date
                    const appointmentDate = document.getElementById('appointmentDate');
                    if (!appointmentDate || !appointmentDate.value.trim()) {
                        showError('appointmentDate', 'Appointment date is required.');
                        isValid = false;
                    } else {
                        // Validate if it's a Thursday
                        const date = new Date(appointmentDate.value);
                        if (date.getDay() !== 4) { // 4 represents Thursday
                            showError('appointmentDate', 'Appointment must be scheduled for a Thursday.');
                            isValid = false;
                        }
                    }

                    // Gender validation
                    if (elements.gender.value === 'Other') {
                        const othergenderInput = document.getElementById('othergender');
                        if (!othergenderInput.value.trim()) {
                            showError('othergender', 'Please specify your gender.');
                            isValid = false;
                        }
                    }

                    // Profile photo validation
                    if (!elements.profilePhoto.src || elements.profilePhoto.src === '') {
                        showError('myFile', 'Please upload a profile photo.');
                        isValid = false;
                    }

                    // ID photo validation
                    if (!elements.idPhoto.src || elements.idPhoto.src === '') {
                        showError('idPhoto', 'Please upload an ID photo.');
                        isValid = false;
                    }

                    if (!isValid) {
                        showError('errorMessage', 'Please fill in all required fields correctly.');
                    }

                } catch (error) {
                    console.error('Form validation error:', error);
                    showError('errorMessage', 'Form validation error occurred.');
                    isValid = false;
                }

                return isValid;
            };

            // Form Submission
            form.addEventListener('submit', e => {
                e.preventDefault();
                
                // Validate form first
                if (!validateForm()) {
                    showError('errorMessage', 'Please fill in all required fields correctly.');
                    return;
                }

                const formData = new FormData(form);
                formData.append('submit_appointment', '1');
                formData.append('csrf_token', elements.csrfToken.value);

                // Show loading spinner
                elements.loadingSpinner.style.display = 'block';
                
                // Clear previous messages
                showError('errorMessage', '');
                showSuccess('successMessage', '');

                fetch(window.location.href, {
                    method: 'POST',
                    body: formData,
                    headers: { 
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    },
                    credentials: 'same-origin'
                })
                .then(async response => {
                    const contentType = response.headers.get('content-type');
                    const responseText = await response.text();
                    
                    console.log('Raw server response:', responseText); // Debug line
                    console.log('Response status:', response.status); // Debug line
                    console.log('Content-Type:', contentType); // Debug line
                    
                    if (!response.ok) {
                        throw new Error(`Server Error (${response.status}): ${responseText}`);
                    }
                    
                    try {
                        return JSON.parse(responseText);
                    } catch (e) {
                        console.error('JSON Parse Error:', e);
                        console.error('Raw Response:', responseText);
                        throw new Error('Invalid JSON response from server');
                    }
                })
                .then(data => {
                    console.log('Parsed response data:', data); // Debug line
                    elements.loadingSpinner.style.display = 'none';

                    if (data.success) {
                        showSuccess('successMessage', data.message || 'Appointment submitted successfully.');
                        // Disable form to prevent double submission
                        form.querySelectorAll('input, select, button').forEach(el => el.disabled = true);
                        // Redirect after showing success message
                        setTimeout(() => {
                            window.location.href = '../dashboard/dashboard.php';
                        }, 2000);
                    } else {
                        // Handle validation errors
                        if (data.errors) {
                            Object.entries(data.errors).forEach(([field, message]) => {
                                showError(field, message);
                            });
                        }
                        if (data.error) {
                            showError('errorMessage', data.error);
                        }
                        if (!data.error && !data.errors) {
                            showError('errorMessage', 'An unexpected error occurred. Please try again.');
                        }
                    }
                })
                .catch(error => {
                    console.error('Full error details:', error); // Enhanced error logging
                    elements.loadingSpinner.style.display = 'none';
                    showError('errorMessage', `Server Error: ${error.message}`);
                });
            });

            // Initialize Feedback
            if (elements.successMessage.textContent) {
                elements.successMessage.style.display = 'block';
            }
            if (elements.errorMessage.textContent) {
                elements.errorMessage.style.display = 'block';
            }
        });
</script>
  <style>
    @import url('https://fonts.googleapis.com/css2?family=Open+Sans:wght@400;500;600;700&display=swap');

        :root {
            --primary-color: #1a3c6e;
            --primary-hover: #0f2452;
            --accent-color: #e6f0fa;
            --text-color: #333333;
            --border-color: #d4d4d4;
            --error-color: #a51c1c;
            --success-color: #2e7d32;
            --bg-color: #f8fafc;
            --white: #ffffff;
            --shadow: 0 2px 6px rgba(0, 0, 0, 0.1);
            --input-bg: #fafafa;
        }

        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Open Sans', sans-serif;
        }

        body {
            margin: 0;
            padding: 0;
            background: var(--bg-color);
        }

        .form-header {
            top: 0;
            z-index: 999;
            background: white;

            display: grid;
            grid-template-columns: auto minmax(0, 1fr) auto;
            align-items: center;

            padding: 1rem 1.5rem;
            border-bottom: 1px solid var(--border-color);
            white-space: nowrap;
            overflow-x: auto;
        }

        .form-container {
            width: 100%;
            max-width: 1200px; /* Optional: limit max width on large screens */
            margin: 0 auto;    /* center container horizontally */
            padding: 2rem;     /* default padding for desktop */
            box-sizing: border-box; /* include padding in width */
        }

        .logo {
            width: 6rem;
            height: auto;
        }

        #form-title {
            margin: 0;
            text-align: center;
            font-size: 2rem;
            font-weight: 700;
            color: var(--text-color);
        }

        .header-right {
            text-align: right;
        }

        .logout-link {
            text-decoration: none;
            font-weight: 600;
            color: var(--primary-color);
            padding: 0.5rem 1rem;
            border: 2px solid var(--primary-color); /* blue border */
            border-radius: 50px; /* rounded edges */
            transition: background 0.2s, color 0.2s, border 0.2s;
        }

        .logout-link:hover {
            background: var(--primary-color);
            color: var(--white);
        }

        h1 {
            font-size: 1.75rem;
            font-weight: 700;
            color: var(--text-color);
        }

        .form-section {
            margin-bottom: 2rem;
            padding-bottom: 1.5rem;
            border-bottom: 1px solid var(--border-color);
        }

        .form-section h2 {
            font-size: 1.25rem;
            font-weight: 600;
            color: var(--primary-color);
            margin-bottom: 1rem;
        }

        .form-group {
            margin-bottom: 1.25rem;
        }

        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1.5rem;
        }

        .label {
            display: block;
            font-size: 0.875rem;
            font-weight: 500;
            color: var(--text-color);
            margin-bottom: 0.5rem;
        }

        .label.required::after {
            content: '*';
            color: var(--error-color);
            font-size: 0.75rem;
            margin-left: 0.25rem;
        }

        input[type="text"],
        input[type="email"],
        input[type="time"],
        input[type="number"],
        input[type="date"],
        select,
        input[type="text"].flatpickr-input {
            width: 100%;
            padding: 0.75rem;
            border: 1px solid var(--border-color);
            border-radius: 6px;
            font-size: 0.875rem;
            color: var(--text-color);
            background: var(--input-bg);
            transition: border-color 0.3s, box-shadow 0.3s;
        }

        input:focus,
        select:focus,
        input[type="text"].flatpickr-input:focus {
            outline: none;
            border-color: var(--primary-color);
            box-shadow: 0 0 0 3px rgba(26, 60, 110, 0.1);
        }

        select {
            appearance: none;
            background: var(--input-bg) url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='12' height='12' fill='%23333' viewBox='0 0 16 16'%3E%3Cpath d='M8 12L2 6h12l-6 6z'/%3E%3C/svg%3E") no-repeat right 0.75rem center;
            padding-right: 2rem;
        }

        input[readonly] {
            background: #f5f5f5;
            color: #666;
            cursor: not-allowed;
        }

        .contact-group {
            display: flex;
            gap: 0.75rem;
            align-items: center;
        }

        .contact-group input[type="text"]:first-child {
            width: 4rem;
            background: #f5f5f5;
            pointer-events: none;
            border-radius: 6px;
        }

        .photo-upload-group {
            display: flex;
            flex-direction: column;
            gap: 0.75rem;
            align-items: flex-start;
        }

        .photo-placeholder {
            position: relative;
            width: 12rem;
            height: 12rem;
            background: var(--accent-color);
            border: 1px solid var(--border-color);
            border-radius: 6px;
            display: flex;
            justify-content: center;
            align-items: center;
            font-size: 0.875rem;
            color: #666;
        }

        .photo-preview {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            object-fit: cover;
            border-radius: 6px;
            display: none;
        }

        .photo-preview.active {
            display: block;
        }

        .photo-upload-note {
            font-size: 0.75rem;
            color: #666;
        }

        .action-buttons {
            display: flex;
            gap: 0.75rem;
            flex-wrap: wrap;
        }

        .btn {
            padding: 0.75rem 1.5rem;
            border: none;
            border-radius: 6px;
            font-size: 0.875rem;
            font-weight: 600;
            cursor: pointer;
            transition: background 0.3s, transform 0.2s;
            text-align: center;
        }

        .btn-primary {
            background: var(--primary-color);
            color: var(--white);
        }

        .btn-danger {
            background: var(--error-color);
            color: var(--white);
        }

        .btn-primary:hover {
            background: var(--primary-hover);
            transform: translateY(-1px);
        }

        .btn-danger:hover {
            background: #8b1a1a;
            transform: translateY(-1px);
        }

        .error-message,
        .success-message {
            padding: 0.75rem;
            border-radius: 6px;
            margin: 0.75rem 0;
            font-size: 0.875rem;
            text-align: center;
            display: none;
        }

        .error-message {
            background: #fef2f2;
            color: var(--error-color);
            border: 1px solid var(--error-color);
        }

        .success-message {
            background: #f0fdf4;
            color: var(--success-color);
            border: 1px solid var(--success-color);
        }

        .error {
            color: var(--error-color);
            font-size: 0.75rem;
            margin-top: 0.375rem;
            display: none;
        }

        .loading-spinner {
            display: none;
            border: 3px solid #e5e7eb;
            border-top: 3px solid var(--primary-color);
            border-radius: 50%;
            width: 1.5rem;
            height: 1.5rem;
            animation: spin 1s linear infinite;
            margin: 0.75rem auto;
        }

        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }

        .progress-bar {
            display: none;
            width: 100%;
            height: 0.25rem;
            background: #e5e7eb;
            border-radius: 4px;
            overflow: hidden;
            margin-top: 0.5rem;
        }

        .progress-bar div {
            height: 100%;
            background: var(--primary-color);
            width: 0;
            transition: width 0.3s ease;
        }

        .flatpickr-day.thursday.available-date {
            color: #03e330 !important; /* ensures text is visible */
            font-weight: 700;
            border-radius: 4px;
        }

        .disabled-date {
            color: #9ca3af; /* gray out past or non-Thursday dates */
            cursor: not-allowed;
        }

        .flatpickr-day:not(.thursday) {
            color: #ccc !important;
            pointer-events: none;
        }

        .flatpickr-day.selected {
            background: var(--primary-color) !important;
            color: var(--white) !important;
            border-color: var(--primary-color) !important;
        }
        .flatpickr-day.full-date{
            color: red !important;
            pointer-events: none;
        }
        .flatpickr-calendar {
            font-size: 0.875rem;
        }

        .debug-log {
            font-size: 0.75rem;
            color: #555;
            background: #f0f0f0;
            padding: 0.75rem;
            border-radius: 6px;
            margin: 0.75rem 0;
            display: none;
        }

        button:focus,
        a:focus,
        input:focus,
        select:focus {
            outline: 2px solid var(--primary-color);
            outline-offset: 2px;
        }

        @media (max-width: 768px) {

            h1 {
                font-size: 1.5rem;
            }

            .form-row {
                grid-template-columns: 1fr;
            }

            .photo-placeholder,
            .photo-preview {
                width: 10rem;
                height: 10rem;
            }
            .btn {
                width: 100%;
            }

            .contact-group {
                flex-direction: column;
            }

            .contact-group input[type="text"]:first-child {
                width: 100%;
            }

            .flatpickr-calendar {
                width: 100% !important;
                max-width: 300px;
            }
        }

            .available-date {
                text-emphasis-color: green;
                color: white;
            }
    
    @media (max-width: 640px) { /* adjust breakpoint as needed */
    .form-header {
        grid-template-columns: 1fr; /* single column */
        grid-template-rows: auto auto auto; /* one row per item */
        row-gap: 0.5rem; /* spacing between rows */
        white-space: normal; /* allow text to wrap */
    }
      
          .form-header > * {
        justify-self: center; /* center */
    }
          .form-container {
        padding: 1rem; /* reduce padding on small screens */
    }

}


        @media (max-width: 480px) {
            .form-container {
                padding: 1rem;
            }

            h1 {
                font-size: 1.25rem;
            }

            .form-section h2 {
                font-size: 1rem;
            }

            .photo-placeholder,
            .photo-preview {
                width: 8rem;
                height: 8rem;
            }
        }
  </style>
</html>