<?php
session_start();

// Prevent caching to ensure fresh content
header("Cache-Control: no-cache, no-store, must-revalidate");
header("Pragma: no-cache");
header("Expires: 0");

// Redirect if not logged in
if (!isset($_SESSION['status_Account']) || !isset($_SESSION['email']) || $_SESSION['status_Account'] !== 'logged_in') {
    header("Location: ../index.php");
    exit();
}

include '../database/db.php';

// Initialize debug log
$debug_log = [];

// Check if database connection is successful
if (!$connection) {
    $debug_log[] = "Database connection failed: " . mysqli_connect_error();
    die("Database connection failed. Please try again later.");
}

try {
    // Fetch user data
    $email = $_SESSION['email'];
    $stmt = $connection->prepare("SELECT user_id FROM data WHERE email = ?");
    if ($stmt === false) {
        $debug_log[] = "User query preparation failed: " . $connection->error;
        throw new Exception("Failed to prepare user query.");
    }
    $stmt->bind_param("s", $email);
    $stmt->execute();
    $result = $stmt->get_result();
    $user = $result->fetch_assoc();
    $user_id = $user['user_id'] ?? null;
    $stmt->close();

    if (!$user_id) {
        $debug_log[] = "No user found for email: $email";
        header("Location: ../index.php");
        exit();
    }

    // Fetch appointment details
    $query = "SELECT appointment_id, first_name, middle_name, last_name, gender, other_gender, birthdate, age, 
                    occupation, address, region, email, contact, appointment_date, appointment_time, purpose, 
                    profile_photo, status, created_at, reference_id 
            FROM appointments 
            WHERE user_id = ?
            ORDER BY created_at DESC"; // ascending

    $stmt = $connection->prepare($query);
    if ($stmt === false) {
        $debug_log[] = "Appointment query preparation failed: " . $connection->error;
        throw new Exception("Failed to prepare appointment query: " . $connection->error);
    }
    $stmt->bind_param("i", $user_id);
    if (!$stmt->execute()) {
        $debug_log[] = "Appointment query execution failed: " . $stmt->error;
        throw new Exception("Failed to execute appointment query.");
    }
    //$result = $stmt->get_result();
    //$appointment = $result->fetch_assoc();
    //$stmt->close();

    $result = $stmt->get_result();

    $appointment = null;

    // Check if an appointment_id was passed in the URL
    if (isset($_GET['appointment_id'])) {
        $requested_id = intval($_GET['appointment_id']);
        while ($row = $result->fetch_assoc()) {
            if ($row['appointment_id'] == $requested_id) {
                $appointment = $row;
                break;
            }
        }

        // Fallback if not found
        if (!$appointment) {
            $appointment = $result->fetch_assoc(); // fallback to latest
        }
    } else {
        $appointment = $result->fetch_assoc(); // latest appointment
    }

    $stmt->close();


    if (!$appointment) {
        $debug_log[] = "No appointment found for user_id: $user_id";
        header("Location: ../fillupform/fillupform.php");
        exit();
    }

    // Fetch ID details from user_information
    $id_type = '';
    $id_number = '';
    $id_photo = '';
    $is_valid_id_photo = false;
    $id_photo_url = '';
    $stmt = $connection->prepare("SELECT id_type, id_number, id_photo FROM user_information WHERE user_id = ?");
    if ($stmt === false) {
        $debug_log[] = "User information query preparation failed: " . $connection->error;
    } else {
        $stmt->bind_param("i", $user_id);
        if ($stmt->execute()) {
            $result = $stmt->get_result();
            $user_info = $result->fetch_assoc();
            $id_type = $user_info['id_type'] ?? '';
            $id_number = $user_info['id_number'] ?? '';
            $id_photo = $user_info['id_photo'] ?? '';
        } else {
            $debug_log[] = "User information query execution failed: " . $stmt->error;
        }
        $stmt->close();
    }

    // Validate profile photo
    $is_valid_photo = false;
    $profile_photo = $appointment['profile_photo'] ?? $_SESSION['profilePhoto'] ?? '';
    $base_server_path = '../ProfileImage/image/';
    $base_relative_url = './image/';

    if ($profile_photo) {
        $profile_photo = ltrim($profile_photo, '/\\');
        $full_photo_path = str_replace('./image/', $base_server_path, $profile_photo);
        $debug_log[] = "Profile photo stored path: '$profile_photo', computed server path: '$full_photo_path'";
        if (file_exists($full_photo_path) && is_file($full_photo_path)) {
            $is_valid_photo = true;
            $profile_photo_url = str_replace('./image/', $base_relative_url, $profile_photo);
        } else {
            $debug_log[] = "Profile photo file not found or invalid: '$full_photo_path'";
        }
    } else {
        $debug_log[] = "Profile photo is empty or not set.";
    }

    // Validate ID photo
    if ($id_photo) {
        $id_photo = ltrim($id_photo, '/\\');
        $full_id_photo_path = str_replace('../image/', $base_server_path, $id_photo);
        $debug_log[] = "ID photo stored path: '$id_photo', computed server path: '$full_id_photo_path'";
        if (file_exists($full_id_photo_path) && is_file($full_id_photo_path)) {
            $is_valid_id_photo = true;
            $id_photo_url = str_replace(['../image/', './image/'], $base_relative_url, $id_photo);
        } else {
            $debug_log[] = "ID photo file not found or invalid: '$full_id_photo_path'";
        }
    } else {
        $debug_log[] = "ID photo is empty or not set.";
    }

} catch (Exception $e) {
    $debug_log[] = "Error: " . $e->getMessage();
} finally {
    if (isset($connection) && $connection) {
        $connection->close();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta http-equiv="Cache-Control" content="no-cache, no-store, must-revalidate">
    <meta http-equiv="Pragma" content="no-cache">
    <meta http-equiv="Expires" content="0">
    <link href="https://unpkg.com/boxicons@2.1.4/css/boxicons.min.css" rel="stylesheet">
    <link rel="icon" type="image/x-icon" href="../image/icons/logo1.ico">
    <link rel="stylesheet" href="../css/style.css">
    <link rel="stylesheet" href="../css/dashboard1.css">
    <title>Dashboard</title>
    
</head>
<body>
    <div class="sidebar">
    <button class="close-sidebar" aria-label="Close Sidebar">
        <i class='bx bx-x'></i>
    </button>
    <img src="../image/icons/logo1.ico" alt="Organization Logo" class="logo">
    <h2>Dashboard</h2>
    
    <a href="dashboard.php" class="<?= basename($_SERVER['PHP_SELF']) == 'dashboard.php' ? 'active' : '' ?>">Dashboard</a>
    <a href="../dashboard/edit_profile/edit_profile.php" class="<?= basename($_SERVER['PHP_SELF']) == 'edit_profile.php' ? 'active' : '' ?>">Edit Profile</a>

    <!-- appointments -->
    <a href="../dashboard/appointments/appointment_table.php" class="<?= basename($_SERVER['PHP_SELF']) == 'appointments.php' ? 'active' : '' ?>">Appointments</a>

    <a href="../logout.php">Logout</a>
    </div>

    <div class="main-content">
        <div class="dashboard-header">
            <button class="menu-toggle" aria-label="Toggle Sidebar" aria-expanded="false">
                <i class='bx bx-menu'></i>
            </button>
            <h1>Welcome to Your Dashboard</h1>
        </div>
        <div class="card">
            <h2>Your Appointment Details</h2>
            <div class="photo-container">
                <?php if ($is_valid_photo && !empty($profile_photo_url)): ?>
                    <img src="<?php echo htmlspecialchars($profile_photo_url); ?>" alt="Profile Photo" class="profile-photo">
                <?php else: ?>
                    <div class="profile-placeholder">No Photo Available</div>
                <?php endif; ?>
                <button class="view-id-btn" aria-label="View ID Details">View ID</button>
            </div>
            <div class="appointment-details">
                <p><strong>Appointment ID: #</strong> <?php echo htmlspecialchars($appointment['appointment_id']); ?></p>
                <p><strong>Name:</strong> <?php echo htmlspecialchars($appointment['first_name'] . ' ' . ($appointment['middle_name'] ? $appointment['middle_name'] . ' ' : '') . $appointment['last_name']); ?></p>
                <p><strong>Gender:</strong> <?php echo htmlspecialchars($appointment['gender'] === 'Other' ? $appointment['other_gender'] : $appointment['gender']); ?></p>
                <p><strong>Date of Birth:</strong> 
                    <?php 
                        $birthdate = $appointment['birthdate'];
                        $formatted_birthdate = date("F j, Y", strtotime($birthdate));
                        echo htmlspecialchars($formatted_birthdate); 
                    ?>
                </p>
                <p><strong>Age:</strong> <?php echo htmlspecialchars($appointment['age']); ?></p>
                <p><strong>Occupation:</strong> <?php echo htmlspecialchars($appointment['occupation']); ?></p>
                <p><strong>Address:</strong> <?php echo htmlspecialchars($appointment['address']); ?></p>
                <p><strong>Region:</strong> <?php echo htmlspecialchars($appointment['region']); ?></p>
                <p><strong>Email:</strong> <?php echo htmlspecialchars($appointment['email']); ?></p>
                <p><strong>Contact Number:</strong> +63<?php echo htmlspecialchars($appointment['contact']); ?></p>
                <p><strong>Appointment Date:</strong> 
                    <?php 
                        $appointment_date = $appointment['appointment_date'];
                        $formatted_date = date("F j, Y", strtotime($appointment_date));
                        echo htmlspecialchars($formatted_date); 
                    ?>
                </p>
                <p><strong>Appointment Time:</strong> 
                    <?php 
                        $time_24hr = $appointment['appointment_time'];
                        $time_12hr = date("g:i A", strtotime($time_24hr));
                        echo htmlspecialchars($time_12hr); 
                    ?>
                </p>
                <p><strong>Purpose:</strong> <?php echo htmlspecialchars($appointment['purpose']); ?></p>
                <?php if ($appointment['status'] === 'Approved' && !empty($appointment['reference_id'])): ?>
                    <p><strong>Reference ID:</strong> <?php echo htmlspecialchars($appointment['reference_id']); ?></p>
                <?php endif; ?>
                <p><strong>Submitted On:</strong> 
                    <?php 
                        $submitted_on = $appointment['created_at'];
                        $formatted_submitted_on = date("F j, Y", strtotime($submitted_on));
                        echo htmlspecialchars($formatted_submitted_on); 
                    ?></p>
                <p><strong>Time:</strong>
                    <?php 
                        $time_on = $appointment['created_at'];
                        $formatted_time_on = date('g:i A', strtotime($time_on));
                        echo htmlspecialchars($formatted_time_on);
                    ?>
                </p>
            </div>
            <div class="status-message status-<?php echo strtolower($appointment['status']); ?>">
                Your appointment is <strong><?php echo htmlspecialchars($appointment['status']); ?></strong>.
                <?php
                switch ($appointment['status']) {
                    case 'Approved':
                        echo 'Please arrive on time for your appointment.';
                        break;
                    case 'Rejected':
                    case 'Cancelled':
                        echo 'Please submit a new appointment or contact support.';
                        break;
                    default:
                        echo 'We are reviewing your appointment. You will be notified soon.';
                }
                ?>
            </div>
        </div>
    </div>

    <!-- Modal for ID Details -->
    <div class="modal" id="idModal">
        <div class="modal-content">
            <div class="modal-header">
                <h3>ID Details</h3>
                <button class="close-btn" aria-label="Close Modal">×</button>
            </div>
            <div class="modal-body">
                <?php if ($is_valid_id_photo && !empty($id_photo_url)): ?>
                    <img src="<?php echo htmlspecialchars($id_photo_url . '?v=' . time()); ?>" alt="ID Photo" class="id-photo">
                <?php else: ?>
                    <div class="id-placeholder">No ID Photo Available</div>
                <?php endif; ?>
                <p class="id-type"><strong>ID Type:</strong> <?php echo htmlspecialchars($id_type ?: 'Not Provided'); ?></p>
                <p class="id-number"><strong>ID Number:</strong> <?php echo htmlspecialchars($id_number ?: 'Not Provided'); ?></p>
            </div>
        </div>
    </div>

<script src="../scripts/dashboard1.js"></script>
</body>
</html>