//flatpicker
const MAX_APPOINTMENTS = 20;
let appointmentCounts = {};

function initFlatpickr() {
    flatpickr('#appointmentDate', {
        dateFormat: 'Y-m-d',
        minDate: 'today', // optional, ensures past dates disabled

        disable: [
            function(date) {
                const today = new Date();
                today.setHours(0, 0, 0, 0); // midnight for proper comparison
                const formatted = flatpickr.formatDate(date, 'Y-m-d');
                const count = appointmentCounts?.[formatted] ?? 0;

                // Disable if:
                // 1. Not Thursday
                // 2. Past date
                // 3. Fully booked
                return date.getDay() !== 4 || date < today || count >= MAX_APPOINTMENTS;
            }
        ],

        onDayCreate: (dObj, dStr, fp, dayElem) => {
            const date = flatpickr.formatDate(dayElem.dateObj, 'Y-m-d');
            const count = appointmentCounts?.[date] ?? 0;
            const today = new Date();
            today.setHours(0,0,0,0);

            if (count >= MAX_APPOINTMENTS) {
                dayElem.classList.add('full-date');
            } else if (dayElem.dateObj.getDay() === 4 && dayElem.dateObj >= today) {
                dayElem.classList.add('thursday', 'available-date');
            } else {
                dayElem.classList.add('disabled-date'); // past or non-Thursday
            }
        }
    });
}

  // Utility Functions
        const debounce = (func, wait) => {
            let timeout;
            return (...args) => {
                clearTimeout(timeout);
                timeout = setTimeout(() => func.apply(this, args), wait);
            };
        };

        const showError = (id, message) => {
            const errorEl = document.getElementById(`${id}-error`);
            if (errorEl) {
                errorEl.textContent = message;
                errorEl.style.display = message ? 'block' : 'none';
                errorEl.setAttribute('aria-live', 'assertive');
            }
        };

        const showSuccess = (id, message) => {
            const successEl = document.getElementById(id);
            if (successEl) {
                successEl.textContent = message;
                successEl.style.display = message ? 'block' : 'none';
                successEl.setAttribute('aria-live', 'assertive');
            }
        };

        // Main Logic
        document.addEventListener('DOMContentLoaded', () => {
            
            //fetch data in database appointment table
            fetch('./get_appointment_counts.php')
                .then(response => response.json())
                .then(data => {
                    appointmentCounts = data;
                    initFlatpickr();
                })
                .catch(error => console.error(error));

            const form = document.getElementById('registrationForm');
            const elements = {
                gender: document.getElementById('gender'),
                othergenderGroup: document.getElementById('othergenderGroup'),
                birthdate: document.getElementById('birthdate'),
                age: document.getElementById('age'),
                contact: document.getElementById('contact'),
                appointmentDate: document.getElementById('appointmentDate'),
                appointmentTime: document.getElementById('appointmentTime'),
                successMessage: document.getElementById('successMessage'),
                errorMessage: document.getElementById('errorMessage'),
                loadingSpinner: document.getElementById('loadingSpinner'),
                idType: document.getElementById('idType'),
                idNumber: document.getElementById('idNumber'),
                idPhotoInput: document.getElementById('idPhotoInput'),
                idPhotoPreview: document.getElementById('idPhotoImg'),
                chooseIdPhotoBtn: document.getElementById('chooseIdPhotoBtn'),
                removeIdPhotoBtn: document.getElementById('removeIdPhotoBtn'),
                idPhotoSpinner: document.getElementById('idPhotoSpinner'),
                idPhotoSuccess: document.getElementById('idPhotoSuccess'),
                idPhotoProgress: document.getElementById('idPhotoProgress'),
                compressedImage: document.getElementById('compressedImage'),
                actionButtons: document.getElementById('actionButtons'),
                region: document.getElementById('region'),
                email: document.getElementById('email'),
                csrfToken: document.querySelector('input[name="csrf_token"]'), // <-- get CSRF token from hidden input
                profilePhoto: document.getElementById('profilePhotoImg'), // get profile photo src
                idPhoto: document.getElementById('idPhotoImg') // get id photo src
            };

            // Initialize debug log visibility
            const debugLog = document.querySelector('.debug-log');
            if (debugLog && debugLog.textContent.trim()) {
                debugLog.style.display = 'block';
            }



            // Toggle Other Gender
            const toggleOtherGender = () => {
                elements.othergenderGroup.style.display = elements.gender.value === 'Other' ? 'block' : 'none';
            };
            elements.gender.addEventListener('change', toggleOtherGender);
            toggleOtherGender();

            // Auto-calculate Age
            elements.birthdate.addEventListener('change', () => {
                const birthdate = new Date(elements.birthdate.value);
                const today = new Date();
                let age = today.getFullYear() - birthdate.getFullYear();
                const monthDiff = today.getMonth() - birthdate.getMonth();
                if (monthDiff < 0 || (monthDiff === 0 && today.getDate() < birthdate.getDate())) {
                    age--;
                }
                elements.age.value = age >= 0 ? age : '';
                validateField('age', elements.age.value);
            });

            // Restrict Contact Input
            elements.contact.addEventListener('input', e => {
                e.target.value = e.target.value.replace(/[^0-9]/g, '').slice(0, 10);
                validateField('contact', e.target.value);
            });

            // Image Compression
            const compressImage = (file, maxSizeMB, targetSize, callback) => {
                const maxSizeBytes = maxSizeMB * 1024 * 1024;
                elements.idPhotoSpinner.style.display = 'block';
                elements.idPhotoProgress.style.display = 'block';
                elements.idPhotoProgress.querySelector('div').style.width = '10%';

                const img = new Image();
                const reader = new FileReader();
                reader.onload = e => {
                    img.src = e.target.result;
                    img.onload = () => {
                        try {
                            const canvas = document.createElement('canvas');
                            const ctx = canvas.getContext('2d');
                            const size = Math.min(img.width, img.height);
                            canvas.width = targetSize;
                            canvas.height = targetSize;
                            const offsetX = (img.width - size) / 2;
                            const offsetY = (img.height - size) / 2;
                            ctx.drawImage(img, offsetX, offsetY, size, size, 0, 0, targetSize, targetSize);

                            let quality = 0.85;
                            let compressedDataUrl;
                            do {
                                compressedDataUrl = canvas.toDataURL('image/jpeg', quality);
                                quality -= 0.05;
                                elements.idPhotoProgress.querySelector('div').style.width = `${100 - (quality * 100)}%`;
                            } while (compressedDataUrl.length / 4 * 3 > maxSizeBytes && quality > 0.1);

                            fetch(compressedDataUrl)
                                .then(res => res.blob())
                                .then(blob => {
                                    const compressedFile = new File([blob], file.name, { type: 'image/jpeg' });
                                    callback(compressedFile, compressedDataUrl);
                                    elements.idPhotoSpinner.style.display = 'none';
                                    elements.idPhotoProgress.style.display = 'none';
                                })
                                .catch(() => {
                                    showError('idPhoto', 'Error compressing image.');
                                    elements.idPhotoSpinner.style.display = 'none';
                                    elements.idPhotoProgress.style.display = 'none';
                                });
                        } catch (error) {
                            console.error('Image compression error:', error);
                            showError('idPhoto', 'Error processing image.');
                            elements.idPhotoSpinner.style.display = 'none';
                            elements.idPhotoProgress.style.display = 'none';
                        }
                    };
                    img.onerror = () => {
                        showError('idPhoto', 'Invalid image file. Please upload a valid JPEG.');
                        elements.idPhotoSpinner.style.display = 'none';
                        elements.idPhotoProgress.style.display = 'none';
                    };
                };
                reader.onerror = () => {
                    showError('idPhoto', 'Error reading file.');
                    elements.idPhotoSpinner.style.display = 'none';
                    elements.idPhotoProgress.style.display = 'none';
                };
                reader.readAsDataURL(file);
            };

            // Add Remove ID Photo Button
            const addRemoveButton = () => {
                const existingBtn = document.getElementById('removeIdPhotoBtn');
                if (existingBtn) {
                    existingBtn.remove();
                }

                const btn = document.createElement('button');
                btn.type = 'button';
                btn.className = 'btn btn-danger';
                btn.id = 'removeIdPhotoBtn';
                btn.textContent = 'Remove ID Photo';
                btn.setAttribute('aria-label', 'Remove uploaded ID photo');
                elements.actionButtons.appendChild(btn);
                btn.addEventListener('click', handleRemoveIdPhoto);
                elements.removeIdPhotoBtn = btn;
            };

            // ID Photo Selection and Auto-Upload
            elements.chooseIdPhotoBtn.addEventListener('click', () => elements.idPhotoInput.click());

            elements.idPhotoInput.addEventListener('change', e => {
                const file = e.target.files[0];
                showError('idPhoto', '');
                showSuccess('idPhotoSuccess', '');

                if (file) {
                    if (!['image/jpeg', 'image/jpg'].includes(file.type)) {
                        showError('idPhoto', 'Please upload a .jpg or .jpeg file.');
                        return;
                    }

                    if (!elements.idType.value) {
                        showError('idType', 'Please select a valid ID type.');
                        return;
                    }
                    if (!elements.idNumber.value || !/^[A-Za-z0-9-]{1,50}$/.test(elements.idNumber.value)) {
                        showError('idNumber', 'ID number must be alphanumeric (up to 50 characters).');
                        return;
                    }

                    compressImage(file, 0.3, 192, (compressedFile, dataUrl) => {
                        elements.idPhotoPreview.src = dataUrl;
                        elements.idPhotoPreview.classList.add('active');
                        elements.idPhotoPreview.setAttribute('aria-hidden', 'false');

                        const dataTransfer = new DataTransfer();
                        dataTransfer.items.add(compressedFile);
                        elements.idPhotoInput.files = dataTransfer.files;
                        elements.compressedImage.value = dataUrl;

                        // Auto-upload
                        const formData = new FormData(form);
                        formData.append('upload_id_photo', '1');
                        formData.append('csrf_token', elements.csrfToken.value); // <-- add CSRF token

                        elements.idPhotoSpinner.style.display = 'block';
                        fetch('', {
                            method: 'POST',
                            body: formData,
                            headers: { 'X-Requested-With': 'XMLHttpRequest' }
                        })
                            .then(response => response.json())
                            .then(data => {
                                elements.idPhotoSpinner.style.display = 'none';
                                if (data.success) {
                                    showSuccess('idPhotoSuccess', data.message);
                                    elements.idPhotoPreview.src = data.idPhoto;
                                    elements.idPhotoPreview.classList.add('active');
                                    elements.idPhotoPreview.setAttribute('aria-hidden', 'false');
                                    addRemoveButton();
                                    // Auto-refresh after successful upload
                                    setTimeout(() => {
                                        window.location.reload();
                                    }, 1000);
                                } else {
                                    showError('idPhoto', data.errors?.idPhoto || 'Failed to upload ID.');
                                    if (data.errors?.idType) showError('idType', data.errors.idType);
                                    if (data.errors?.idNumber) showError('idNumber', data.errors.idNumber);
                                    elements.idPhotoPreview.src = '';
                                    elements.idPhotoPreview.classList.remove('active');
                                    elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                                    elements.idPhotoInput.value = '';
                                    elements.compressedImage.value = '';
                                }
                            })
                            .catch(error => {
                                console.error('Upload error:', error);
                                elements.idPhotoSpinner.style.display = 'none';
                                showError('idPhoto', 'Network error.');
                                elements.idPhotoPreview.src = '';
                                elements.idPhotoPreview.classList.remove('active');
                                elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                                elements.idPhotoInput.value = '';
                                elements.compressedImage.value = '';
                            });
                    });
                }
            });

            // Remove ID Photo
            const handleRemoveIdPhoto = () => {
                const formData = new FormData();
                formData.append('removeIdPhoto', 'true');
                formData.append('csrf_token', elements.csrfToken.value); // <-- add CSRF token
                elements.idPhotoSpinner.style.display = 'block';
                fetch('', {
                    method: 'POST',
                    body: formData,
                    headers: { 'X-Requested-With': 'XMLHttpRequest' }
                })
                    .then(response => response.json())
                    .then(data => {
                        elements.idPhotoSpinner.style.display = 'none';
                        if (data.success) {
                            showSuccess('idPhotoSuccess', data.message);
                            elements.idPhotoPreview.src = '';
                            elements.idPhotoPreview.classList.remove('active');
                            elements.idPhotoPreview.setAttribute('aria-hidden', 'true');
                            elements.idPhotoInput.value = '';
                            elements.compressedImage.value = '';
                            elements.idType.value = '';
                            elements.idNumber.value = '';
                            const removeBtn = document.getElementById('removeIdPhotoBtn');
                            if (removeBtn) removeBtn.remove();
                            elements.removeIdPhotoBtn = null;
                        } else {
                            showError('idPhoto', 'Error removing ID photo.');
                        }
                    })
                    .catch(error => {
                        console.error('Remove photo error:', error);
                        elements.idPhotoSpinner.style.display = 'none';
                        showError('idPhoto', 'Network error.');
                    });
            };

            if (elements.removeIdPhotoBtn) {
                elements.removeIdPhotoBtn.addEventListener('click', handleRemoveIdPhoto);
            }

            // Real-time Validation
            const validateField = (id, value) => {
                const fieldLabels = {
                    lastName: 'Last name',
                    firstName: 'First name',
                    occupation: 'Occupation',
                    address: 'Address',
                    purpose: 'Purpose',
                    region: 'Region',
                    email: 'Email',
                    contact: 'Contact number',
                    age: 'Age',
                    idNumber: 'ID number',
                    gender: 'Gender',
                    idType: 'ID type'
                };

                try {
                    switch (id) {
                        case 'lastName':
                        case 'firstName':
                        case 'occupation':
                        case 'address':
                        case 'purpose':
                            showError(id, value.trim() ? '' : `${fieldLabels[id]} is required.`);
                            break;
                        case 'region':
                            showError(id, value.trim() ? '' : 'Please select a valid region.');
                            break;
                        case 'email':
                            showError(id, /^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(value) ? '' : 'Please enter a valid email.');
                            break;
                        case 'contact':
                            showError(id, /^[0-9]{10}$/.test(value) ? '' : 'Please enter a valid 10-digit number.');
                            break;
                        case 'age':
                            showError(id, (value >= 1 && value <= 120) ? '' : 'Age must be between 1 and 120.');
                            break;
                        case 'idNumber':
                            showError(id, /^[A-Za-z0-9-]{1,50}$/.test(value) ? '' : 'ID number must be alphanumeric (up to 50 characters).');
                            break;
                        case 'gender':
                            showError(id, value ? '' : 'Please select a gender.');
                            if (value === 'Other') {
                                const otherValue = document.getElementById('othergender').value;
                                showError('othergender', otherValue.trim() ? '' : 'Please specify your gender.');
                            }
                            break;
                        case 'idType':
                            showError(id, value ? '' : 'Please select a valid ID type.');
                            break;
                    }
                } catch (error) {
                    console.error(`Validation error for ${id}:`, error);
                    showError(id, 'Validation error occurred.');
                }
            };

            ['lastName', 'firstName', 'occupation', 'address', 'region', 'purpose', 'contact', 'age', 'idNumber', 'gender', 'idType'].forEach(id => {
                const input = document.getElementById(id);
                if (input) {
                    input.addEventListener('input', debounce(e => validateField(id, e.target.value), 300));
                }
            });

            // Form Validation
            const validateForm = () => {
                let isValid = true;
                const fields = [
                    { id: 'lastName', message: 'Last name is required.' },
                    { id: 'firstName', message: 'First name is required.' },
                    { id: 'gender', message: 'Please select a gender.' },
                    { id: 'birthdate', message: 'Date of birth is required.' },
                    { id: 'age', message: 'Age is required.' },
                    { id: 'occupation', message: 'Occupation is required.' },
                    { id: 'address', message: 'Address is required.' },
                    { id: 'region', message: 'Please select a valid region.' },
                    { id: 'email', message: 'Email is required.' },
                    { id: 'contact', message: 'Contact number is required.' },
                    { id: 'purpose', message: 'Purpose is required.' },
                    { id: 'idType', message: 'ID type is required.' },
                    { id: 'idNumber', message: 'ID number is required.' }
                ];

                try {
                    // Clear all previous errors first
                    fields.forEach(field => {
                        showError(field.id, '');
                    });
                    showError('appointmentDate', '');
                    showError('errorMessage', '');

                    // Validate all fields
                    fields.forEach(field => {
                        const input = document.getElementById(field.id);
                        if (!input || !input.value.trim()) {
                            showError(field.id, field.message);
                            isValid = false;
                        } else {
                            validateField(field.id, input.value);
                            if (document.getElementById(`${field.id}-error`).style.display === 'block') {
                                isValid = false;
                            }
                        }
                    });

                    // Special validation for appointment date
                    const appointmentDate = document.getElementById('appointmentDate');
                    if (!appointmentDate || !appointmentDate.value.trim()) {
                        showError('appointmentDate', 'Appointment date is required.');
                        isValid = false;
                    } else {
                        // Validate if it's a Thursday
                        const date = new Date(appointmentDate.value);
                        if (date.getDay() !== 4) { // 4 represents Thursday
                            showError('appointmentDate', 'Appointment must be scheduled for a Thursday.');
                            isValid = false;
                        }
                    }

                    // Gender validation
                    if (elements.gender.value === 'Other') {
                        const othergenderInput = document.getElementById('othergender');
                        if (!othergenderInput.value.trim()) {
                            showError('othergender', 'Please specify your gender.');
                            isValid = false;
                        }
                    }

                    // Profile photo validation
                    if (!elements.profilePhoto.src || elements.profilePhoto.src === '') {
                        showError('myFile', 'Please upload a profile photo.');
                        isValid = false;
                    }

                    // ID photo validation
                    if (!elements.idPhoto.src || elements.idPhoto.src === '') {
                        showError('idPhoto', 'Please upload an ID photo.');
                        isValid = false;
                    }

                    if (!isValid) {
                        showError('errorMessage', 'Please fill in all required fields correctly.');
                    }

                } catch (error) {
                    console.error('Form validation error:', error);
                    showError('errorMessage', 'Form validation error occurred.');
                    isValid = false;
                }

                return isValid;
            };

            // Form Submission
            form.addEventListener('submit', e => {
                e.preventDefault();
                
                // Validate form first
                if (!validateForm()) {
                    showError('errorMessage', 'Please fill in all required fields correctly.');
                    return;
                }

                const formData = new FormData(form);
                formData.append('submit_appointment', '1');
                formData.append('csrf_token', elements.csrfToken.value);

                // Show loading spinner
                elements.loadingSpinner.style.display = 'block';
                
                // Clear previous messages
                showError('errorMessage', '');
                showSuccess('successMessage', '');

                fetch(window.location.href, {
                    method: 'POST',
                    body: formData,
                    headers: { 
                        'X-Requested-With': 'XMLHttpRequest',
                        'Accept': 'application/json'
                    },
                    credentials: 'same-origin'
                })
                .then(async response => {
                    const contentType = response.headers.get('content-type');
                    const responseText = await response.text();
                    
                    console.log('Raw server response:', responseText); // Debug line
                    console.log('Response status:', response.status); // Debug line
                    console.log('Content-Type:', contentType); // Debug line
                    
                    if (!response.ok) {
                        throw new Error(`Server Error (${response.status}): ${responseText}`);
                    }
                    
                    try {
                        return JSON.parse(responseText);
                    } catch (e) {
                        console.error('JSON Parse Error:', e);
                        console.error('Raw Response:', responseText);
                        throw new Error('Invalid JSON response from server');
                    }
                })
                .then(data => {
                    console.log('Parsed response data:', data); // Debug line
                    elements.loadingSpinner.style.display = 'none';

                    if (data.success) {
                        showSuccess('successMessage', data.message || 'Appointment submitted successfully.');
                        // Disable form to prevent double submission
                        form.querySelectorAll('input, select, button').forEach(el => el.disabled = true);
                        // Redirect after showing success message
                        setTimeout(() => {
                            window.location.href = '../dashboard/dashboard.php';
                        }, 2000);
                    } else {
                        // Handle validation errors
                        if (data.errors) {
                            Object.entries(data.errors).forEach(([field, message]) => {
                                showError(field, message);
                            });
                        }
                        if (data.error) {
                            showError('errorMessage', data.error);
                        }
                        if (!data.error && !data.errors) {
                            showError('errorMessage', 'An unexpected error occurred. Please try again.');
                        }
                    }
                })
                .catch(error => {
                    console.error('Full error details:', error); // Enhanced error logging
                    elements.loadingSpinner.style.display = 'none';
                    showError('errorMessage', `Server Error: ${error.message}`);
                });
            });

            // Initialize Feedback
            if (elements.successMessage.textContent) {
                elements.successMessage.style.display = 'block';
            }
            if (elements.errorMessage.textContent) {
                elements.errorMessage.style.display = 'block';
            }
        });