<?php
require '../vendor/autoload.php';
include '../database/db.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;

function generateExcelReport($month, $type) {
    global $connection;

    // Ensure no output has been sent before
    if (ob_get_length()) ob_end_clean();
    
    // Create new Spreadsheet object
    $spreadsheet = new Spreadsheet();
    $sheet = $spreadsheet->getActiveSheet();
    
    // Set title
    $sheet->setCellValue('A1', 'Appointment Requests Report - ' . date('F Y', strtotime($month)));
    $sheet->mergeCells('A1:G1');
    
    // Set headers
    $sheet->setCellValue('A2', 'Client Name');
    $sheet->setCellValue('B2', 'Email');
    $sheet->setCellValue('C2', 'Date');
    $sheet->setCellValue('D2', 'Time');
    $sheet->setCellValue('E2', 'Status');
    $sheet->setCellValue('F2', 'Purpose');
    $sheet->setCellValue('G2', 'Remarks');

    // Style headers
    $headerStyle = [
        'font' => [
            'bold' => true,
            'color' => ['rgb' => 'FFFFFF']
        ],
        'fill' => [
            'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
            'startColor' => ['rgb' => '003087']
        ],
        'alignment' => [
            'horizontal' => \PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER
        ]
    ];
    
    $sheet->getStyle('A1:G1')->applyFromArray($headerStyle);
    $sheet->getStyle('A2:G2')->applyFromArray($headerStyle);

    // Get data from database
    $start_date = date('Y-m-01', strtotime($month));
    $end_date = date('Y-m-t', strtotime($month));
    
    $sql = "SELECT * FROM appointments WHERE DATE(appointment_date) BETWEEN ? AND ?";
    if ($type !== 'all') {
        $sql .= " AND status = ?";
    }
    $sql .= " ORDER BY created_at DESC";

    $stmt = $connection->prepare($sql);
    
    if ($type !== 'all') {
        $stmt->bind_param("sss", $start_date, $end_date, $type);
    } else {
        $stmt->bind_param("ss", $start_date, $end_date);
    }
    
    $stmt->execute();
    $result = $stmt->get_result();

    $row = 3; // Start from row 3 since we have title and headers
    while ($data = $result->fetch_assoc()) {
        $sheet->setCellValue('A' . $row, $data['first_name'] . ' ' . $data['last_name']);
        $sheet->setCellValue('B' . $row, $data['email']);
        $sheet->setCellValue('C' . $row, date("F j, Y", strtotime($data['appointment_date'])));
        $sheet->setCellValue('D' . $row, date("g:i A", strtotime($data['appointment_time'])));
        $sheet->setCellValue('E' . $row, $data['status']);
        $sheet->setCellValue('F' . $row, $data['purpose']);
        $sheet->setCellValue('G' . $row, $data['remarks']);
        $row++;
    }

    // Auto size columns
    foreach(range('A','G') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }

    // Create Excel file
    $writer = new Xlsx($spreadsheet);
    
    // Set headers for download
    header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
    header('Content-Disposition: attachment;filename="Appointments_Report_' . date('F_Y', strtotime($month)) . '.xlsx"');
    header('Cache-Control: max-age=0');
    header('Cache-Control: max-age=1');
    header('Expires: Mon, 26 Jul 1997 05:00:00 GMT');
    header('Last-Modified: ' . gmdate('D, d M Y H:i:s') . ' GMT');
    header('Cache-Control: cache, must-revalidate');
    header('Pragma: public');

    // Save file to PHP output stream
    $writer->save('php://output');
    exit;
}